/* Copyright (C)
* 2017 - Christoph van Wüllen, DL1YCF
*
*   This program is free software: you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation, either version 3 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program.  If not, see <https://www.gnu.org/licenses/>.
*
*/

/*
 * This is the "ramp" with length 200 which is
 * the step response derived from a digital filter
 * with a Blackman-Harris window.
 * It is used to shape the rising and falling
 * edge of a CW pulse. The goal is to have
 * a pulse whose off-center frequency components
 * quickly fall off.
 *
 * The Blackman-Harris window function is
 *
 * B(x) = a0 - a1 Cos(2 Pi x) + a2 Cos(4 Pi x) - a3 Cos(6 Pi x)
 *
 * with coefficients
 * a0 = 0.35875
 * a1 = 0.48829
 * a2 = 0.14128
 * a3 = 0.01168
 *
 *
 * With my ANAN-7000 and the new protocol, I had problems if the ramp goes
 * from zero to unity, and these problems vanish if the ramp only goes up
 * to 0.99. Then, the 48kHz ramp with 201 samples is
 *
 * cwramp48[i] = 0.99 (Integral of B(x) from zero to i/200) / a0;
 * for i=0,1,2,...,RAMPLEN.
 *
 * Since a0 is the Integral of (Bx) from zero to one, it follows that
 * cwramp[0] = 0.0 and  cwramp[RAMPLEN] = 0.99, and the ramp rises with
 * sigmoidal shape.
 *
 * For the new protocol, we must generate a ramp that is exactly four times
 * as long (804 samples), and where the first four samples are 0.0 and the
 * last four samples are 0.99. Therefore the definition of the "192kHz ramp" is
 *
 * cwramp192[0]=cwramp192[1]=cwramp192[2]=0
 *
 * cwramp192[i] = 0.99 (Integral of B(x) from zero to (i-3)/797) / a0
 * for i=3,4,5,...,4*RAMPLEN.
 *
 * cwramp192[4*RAMPLEN+1]=cwramp192[4*RAMPLEN+2]=cwramp192[4*RAMPLEN+3]=0.99
 *
 * The values here are calculated using MATHEMATICA with 32-digit accuracy
 * (more than we need for double) and copy/pasted into this file.
 *
 * MEASURING SPECTRAL POLLUTION
 * ============================
 *
 * Using an SDR, a DummyLoad/Attenuator and my Kenwood TRX for receiving
 * the signal, I started a 20 wpm CQ call on the SDR at 7010 kHz and
 * monitored the S-meter of my Kenwood rig for different frequencies.
 * I used a 60dB attenuator which made a S9+50dB signal in my Kenwood
 * out of the 10 Watts output of my SDR.
 * Three cases were measured:
 * a) "hard" CW signal, rectangular CW pulses
 * b) CW signal with linear ramps, width 4 msec (RAMPLEN=200)
 * c) CW signal with Blackman-Harris ramp (as defined here), width 4 msec
 *
 * These are the results:
 *
 *  Freq / kHz    S-Meter    S-Meter    S-Meter
 *                 "hard"     "linear"   "Blackman-Harris"
 *  ------------------------------------------------------
 *  7010.0        S9+50       S9+50     S9+50
 *  7009.5        S9+25       S9        S9+10
 *  7009.0        S9+20       S6        S0
 *  7008.5        S9+15       S4        S0
 *  7008.0        S9+10       S3        S0
 *  7005.0        S9+5        S0        S0
 *  7000.0        S8          S0        S0
 *  -----------------------------------------------------
 *
 * (for frequencies above 7010 kHz, one gets mirror-image results).
 *
 * The linear ramp is a big improvement, but the Blackman-Harris
 * ramp does far better if more than 500 Hz off.
 *
 * MATHEMATICAL NOTE:
 * ==================
 *
 * You can combine the better close-by attenuation of the linear ramp
 * with still acceptable far-off attenuation if you use a "raised cosine"
 * profile. Whether "Blackman-Harris" or "raised cosine" is better
 * is to some extent a matter of taste. If you want to go for "raised cosine",
 * this file is the only one to be changed.
 */

// cwramp48 for RAMPLEN=250
double  cwramp48[251] = {
  0.0,
  6.9520488994810607654371035203920775863611193507e-7,
  1.58856887099813472920279063141948676779287726137e-6,
  2.8818893121655845940833946016315580379001434540e-6,
  4.7842400001124923927287109239039191089612829348e-6,
  7.5156089308585214721784152459927499835515938228e-6,
  0.0000113105354009108148012567072317593854118553194432,
  0.0000164217458326801362647699895987208660173358199665,
  0.0000231237874884165637870289024524593191520263866387,
  0.0000317166588805761782172818166624716753936997954378,
  0.0000425294352774586338742897535674682048783876751577,
  0.0000559238872346170705607073920369833513076011802337,
  0.0000722980895589416868824706466098035087018629062041,
  0.0000920900175379576685743488158766248336041793309686,
  0.0001157811266467248575649829218744202990988304449093,
  0.0001438999112841823324456272460916293598761627889148,
  0.0001770254373956461749574670192151549670651580778266,
  0.0002157908431146035798065464277660931510432705513295,
  0.0002608868008114311338310879047869154309798326790950,
  0.0003130649331759561914031946682847871611951351530743,
  0.0003731411751918662187915168519946254531006630583467,
  0.0004419990730910253865075815666434827883914986550044,
  0.0005205930106120913993663888827552814583339778711436,
  0.0006099513521378375483293353006609314168062668331011,
  0.0007111794915567124464038116596482478577330863803654,
  0.0008254627949938378625737563910860247780224164558608,
  0.0009540694248922156302041152155559864730731306259483,
  0.0010983530323036365218952207429234816350296782367241,
  0.0012597553036777424644508168933834426725305574601125,
  0.0014398083479237607811063616309180409938678737352640,
  0.0016401369090692162134231101468216897662212208055006,
  0.0018624603894597357871741844083651730965694056057632,
  0.0021085946681398417459336053298778436231508259982432,
  0.002380453698831929913987854853057381723215531736323,
  0.002680050871794567218036669082838805779388468845650,
  0.003009500123796447082917231758753008239929269667263,
  0.003371016780492935233896960738218310716183815114461,
  0.003766918115641693719806858650365446380546896826085,
  0.004199623611845382482905593278066866079002202894161,
  0.004671654907865295471524056776542690466778396720464,
  0.005185635418011753738391507414110893552545238203795,
  0.005744289609686264679349487213030783723636624501847,
  0.006350441925827313874744698494413250439985725639417,
  0.007007015339795956952579523316261737848816444996481,
  0.007717029531128212155319751135031631379620305463348,
  0.008483598671577020073014387901671870574705055243068,
  0.009309928811964947140037830438627519856577506058276,
  0.010199314861566891650248260749976598728606105717334,
  0.011155137153036157074832071949829691290943061431033,
  0.012180857587273075857949421093553075511693359349657,
  0.013280015354107939270051084445620489691296565382291,
  0.014456222226223730645476277640499107673662228288376,
  0.015713157425372883828050163654627966982266465322408,
  0.017054562061639235561784623541422617192817960692424,
  0.018484233148254207624223222403613339988513268569837,
  0.020006017196287232776741676774490076954913215405338,
  0.021623803395386251062393216272986588568458103600573,
  0.023341516388636046107700169213693916834829088554926,
  0.025163108651521179384809419820724238640191036400399,
  0.027092552486916893486552082772417440636871865420356,
  0.029133831649975887559671126220437581899826940464199,
  0.031290932618721379459452104082175048945308770708992,
  0.033567835528087239743156170398441121720343903553779,
  0.035968504787053966466909176303836154470102475169387,
  0.038496879400404551462196084502670934778601177437968,
  0.041156863018456540289235905333633025766819590673786,
  0.043952313739905526247300695195691021472857486216118,
  0.046887033694630763206194091809157042334944487074534,
  0.049964758434955512539167474140523197689505988693780,
  0.053189146165413353704933323245720897042466050571731,
  0.056563766842537457919972206923741308168009387691919,
  0.060092091177553551523998147003732442439551965815719,
  0.063777479576110165618532257734035526251052210522758,
  0.067623171050313403256113261231126383037436085683050,
  0.071632272139339963601589418363631336741173076459529,
  0.075807745875774187803515058352398892971163086028467,
  0.080152400835545658144178197050313299059031805091959,
  0.084668880309927238117982310634991330046873852234803,
  0.089359651638483897245058670674769792688754349084734,
  0.094226995742135430941957303068960104765254881385321,
  0.099272996895607205206201818432287844527000818200058,
  0.104499532778489032147622766312920634636561285286040,
  0.109908264843900705950660496560555480769552075848083,
  0.115500629043371897694474928413081852124815985391236,
  0.121277826945983144038909631091124607173921149186695,
  0.127240817289083528496480049246837397122748716219095,
  0.133390307997000149280827797220331459827327799073190,
  0.139726748703086247612278105296298068647233843673426,
  0.146250323809221435636218722868824768082172743275938,
  0.152960946115482156759896652045843659446467277088000,
  0.159858251051147508436876745207441148506120122506517,
  0.166941591536499851114734066206498045649199046951237,
  0.174210033503027008826467742367664372511613643422965,
  0.181662352097639903769933353274617668922063712728360,
  0.189297028594393474033214696760410857375563923273557,
  0.197112248034947731376102819093021230104288654803005,
  0.205105897616638536139053848523937477196566861153335,
  0.213275565844553451430578529752067575877911492287444,
  0.221618542461436839260842138778027253143898489199687,
  0.230131819166590679373896933410829728417156835044503,
  0.238812091132204431398041747541828028726589381639198,
  0.247655759322750076480293533153531657047429329993848,
  0.256658933620229114372894844282597557965807292431052,
  0.265817436755168941934448083566019925855661737168900,
  0.275126809040349163162475727848027753145522137358839,
  0.284582313901306660150903351804957030974096943046913,
  0.294178944194734524311339974915224140801314269816888,
  0.303911429302967134478547937387586769070033733349949,
  0.313774242989844726752615992138900924992256306275344,
  0.323761612000388646129314188830448917060570106237290,
  0.333867525383905915981722836647546991953960032570955,
  0.344085744517391455701407897348174317616629301919239,
  0.354409813803420636689085675101840708932129179710116,
  0.364833072014136017196601438826950735724074085999156,
  0.375348664250441808194963690534004415465811606304775,
  0.385949554483139252957715748425865750235982232423658,
  0.396628538640476538834688137959057236513386973738400,
  0.407378258204458460822462145029216098362043759501696,
  0.418191214276273604023595539845123211272217080473451,
  0.429059782069359458038460432807280123529469505166999,
  0.439976225786947097280861204803373600672111988255292,
  0.450932713839414614812557936129502220531179290649862,
  0.461921334355439382691610840951362427651898718150945,
  0.472934110939779633835434934846127497227661177955377,
  0.483963018629541197726389082242589700933198772878013,
  0.495000000000000000000000000000000000000000000000000,
  0.506036981370458802273610917757410299066801227121987,
  0.517065889060220366164565065153872502772338822044623,
  0.528078665644560617308389159048637572348101281849055,
  0.539067286160585385187442063870497779468820709350138,
  0.550023774213052902719138795196626399327888011744708,
  0.560940217930640541961539567192719876470530494833001,
  0.571808785723726395976404460154876788727782919526549,
  0.582621741795541539177537854970783901637956240498304,
  0.593371461359523461165311862040942763486613026261600,
  0.604050445516860747042284251574134249764017767576342,
  0.614651335749558191805036309465995584534188393695225,
  0.625166927985863982803398561173049264275925914000844,
  0.635590186196579363310914324898159291067870820289884,
  0.645914255482608544298592102651825682383370698080761,
  0.656132474616094084018277163352453008046039967429045,
  0.666238387999611353870685811169551082939429893762710,
  0.676225757010155273247384007861099075007743693724656,
  0.686088570697032865521452062612413230929966266650051,
  0.695821055805265475688660025084775859198685730183112,
  0.705417686098693339849096648195042969025903056953087,
  0.714873190959650836837524272151972246854477862641161,
  0.724182563244831058065551916433980074144338262831100,
  0.733341066379770885627105155717402442034192707568948,
  0.742344240677249923519706466846468342952570670006152,
  0.751187908867795568601958252458171971273410618360802,
  0.759868180833409320626103066589170271582843164955497,
  0.768381457538563160739157861221972746856101510800313,
  0.776724434155446548569421470247932424122088507712556,
  0.784894102383361463860946151476062522803433138846665,
  0.792887751965052268623897180906978769895711345196995,
  0.800702971405606525966785303239589142624436076726443,
  0.808337647902360096230066646725382331077936287271640,
  0.815789966496972991173532257632335627488386356577035,
  0.823058408463500148885265933793501954350800953048763,
  0.830141748948852491563123254792558851493879877493483,
  0.837039053884517843240103347954156340553532722912000,
  0.843749676190778564363781277131175231917827256724062,
  0.850273251296913752387721894703701931352766156326574,
  0.856609692002999850719172202779668540172672200926810,
  0.862759182710916471503519950753162602877251283780905,
  0.868722173054016855961090368908875392826078850813305,
  0.874499370956628102305525071586918147875184014608764,
  0.880091735156099294049339503439444519230447924151917,
  0.885500467221510967852377233687079365363438714713960,
  0.890727003104392794793798181567712155472999181799942,
  0.895773004257864569058042696931039895234745118614679,
  0.900640348361516102754941329325230207311245650915266,
  0.905331119690072761882017689365008669953126147765197,
  0.909847599164454341855821802949686700940968194908041,
  0.914192254124225812196484941647601107028836913971533,
  0.918367727860660036398410581636368663258826923540471,
  0.922376828949686596743886738768873616962563914316950,
  0.926222520423889834381467742265964473748947789477242,
  0.929907908822446448476001852996267557560448034184281,
  0.933436233157462542080027793076258691831990612308081,
  0.936810853834586646295066676754279102957533949428269,
  0.940035241565044487460832525859476802310494011306220,
  0.943112966305369236793805908190842957665055512925466,
  0.946047686260094473752699304804308978527142513783882,
  0.948843136981543459710764094666366974233180409326214,
  0.951503120599595448537803915497329065221398822562032,
  0.954031495212946033533090823696163845529897524830613,
  0.956432164471912760256843829601558878279656096446221,
  0.958709067381278620540547895917824951054691229291008,
  0.960866168350024112440328873779562418100173059535801,
  0.962907447513083106513447917227582559363128134579644,
  0.964836891348478820615190580179275761359808963599601,
  0.966658483611363953892299830786306083165170911445074,
  0.968376196604613748937606783727013411431541896399427,
  0.969993982803712767223258323225509923045086784594662,
  0.971515766851745792375776777596386660011486731430163,
  0.972945437938360764438215376458577382807182039307576,
  0.974286842574627116171949836345372033017733534677592,
  0.975543777773776269354523722359500892326337771711624,
  0.976719984645892060729948915554379510308703434617709,
  0.977819142412726924142050578906446924488306640650343,
  0.978844862846963842925167928050170308709056938568967,
  0.979800685138433108349751739250023401271393894282666,
  0.980690071188035052859962169561372480143422493941724,
  0.981516401328422979926985612098328129425294944756932,
  0.982282970468871787844680248864968368620379694536652,
  0.982992984660204043047420476683738262151183555003519,
  0.983649558074172686125255301505586749560014274360583,
  0.984255710390313735320650512786969216276363375498153,
  0.984814364581988246261608492585889106447454761796205,
  0.985328345092134704528475943223457309533221603279536,
  0.985800376388154617517094406721933133920997797105839,
  0.986233081884358306280193141349634553619453103173915,
  0.986628983219507064766103039261781689283816184885539,
  0.986990499876203552917082768241246991760070730332737,
  0.987319949128205432781963330917161194220611531154350,
  0.987619546301168070086012145146942618276784468263677,
  0.987891405331860158254066394670122156376849174001757,
  0.988137539610540264212825815591634826903430594394237,
  0.988359863090930783786576889853178310233778779194499,
  0.988560191652076239218893638369081959006132126264736,
  0.988740244696322257535549183106616557327469442539888,
  0.988901646967696363478104779257076518364970321763276,
  0.989045930575107784369795884784444013526926869374052,
  0.989174537205006162137426243608913975221977583544139,
  0.989288820508443287553596188340351752142266913619635,
  0.989390048647862162451670664699339068583193733166899,
  0.989479406989387908600633611117244718541666022128856,
  0.989558000926908974613492418433356517211608501344996,
  0.989626858824808133781208483148005374546899336941653,
  0.989686935066824043808596805331715212838804864846926,
  0.989739113199188568866168912095213084569020167320905,
  0.989784209156885396420193453572233906848956729448671,
  0.989822974562604353825042532980784845032934841922173,
  0.989856100088715817667554372753908370640123837211085,
  0.989884218873353275142435017078125579700901169555091,
  0.989907909982462042331425651184123375166395820669031,
  0.989927701910441058313117529353390196491298137093796,
  0.989944076112765382929439292607963016648692398819766,
  0.989957470564722541366125710246432531795121612324842,
  0.989968283341119423821782718183337528324606300204562,
  0.989976876212511583436212971097547540680847973613361,
  0.989983578254167319863735230010401279133982664180034,
  0.989988689464599089185198743292768240614588144680557,
  0.989992484391069141478527821584754007250016448406177,
  0.989995215759999887507607271289076096080891038717065,
  0.989997118110687834415405916605398368441962099856546,
  0.989998411431129001865270797209368580513232207122739,
  0.989999304795110051893923456289647960792241363888065,
  0.990000000000000000000000000000000000000000000000000
};

// cwramp192 for RAMPLEN=250
double cwramp192[1004] = {
  0.0,
  0.0,
  0.0,
  0.0,
  1.6659148386951055668096113349261983864472634190e-7,
  3.3629379858743361118638916631197623051677374075e-7,
  5.1222138182304850758845099594055519149915286619e-7,
  6.9749588488050366980701497971378242996511679157e-7,
  8.9524977949880764335827366864596944084370345321e-7,
  1.10862996463009858211065135886647840939649633986e-6,
  1.34080137318763651387480831523827976447293390190e-6,
  1.59495057875383634380691098875643621029338075612e-6,
  1.87428940223725270402692181061568568124384144913e-6,
  2.1820585184657411761957951681391106325576096408e-6,
  2.5215310627010549939645614715695355075043216119e-6,
  2.8960162370578931150038423699037782189046298741e-6,
  3.3088629168078957260093088260268626788234857975e-6,
  3.7634632565462881439002084939986954094846021463e-6,
  4.2632562961958051840584388321665228518047741309e-6,
  4.8117315668191870093921228182793864073898830879e-6,
  5.4124326962079260249035429185064742535040099734e-6,
  6.0689610142110644583787570057198955877932592013e-6,
  6.7849791577636959295198996969344124934921471216e-6,
  7.5642146755704137608327822247962175664849217196e-6,
  8.4104636323942763692765964410648522780364291714e-6,
  9.3275942128969282844243676537003124055562599709e-6,
  0.0000103195503249703267929272179832423002997796263666,
  0.0000113903552024950816755091186720586712781715516379,
  0.0000125441150074547218843245360546037686848223002562,
  0.0000137850224313292613445762099983371892751212580444,
  0.0000151173602956852495293567872540892684410092812922,
  0.0000165455051518730643592398316285808030399439745608,
  0.0000180739308797355387592878400336311198636242629419,
  0.0000197072122852251114381289604802565681581673868298,
  0.0000214500286968195608385494515316327016189629258690,
  0.0000233071675606190225768097523221530294677605592737,
  0.0000252835280339994089953814806096391742372271846661,
  0.0000273841245776895487827642259945722190074905162013,
  0.0000296140905461313491695179294563751545647809608623,
  0.0000319786817759740573182500924525485051587690928981,
  0.0000344832801725452656334234687697508410217896675880,
  0.0000371333972941326723888483412596685180816044253571,
  0.0000399346779339017789870137552821904825971376470428,
  0.0000428929036992656831195540891561499599706951146046,
  0.0000460139965885139179988626968733126235829555496299,
  0.0000493040225644978966940107774938547343659969465592,
  0.0000527691951251609525546289915806369508036663992870,
  0.0000564158788706912269751301794173614553608034487489,
  0.0000602505930670657496732466955505730125471755673815,
  0.0000642800152057439896675562893106167185268286913199,
  0.0000685109845592589327740357497966844932168043037728,
  0.0000729505057324433693343083494898699869513265812357,
  0.0000776057522090185597654400971098466405327817312274,
  0.0000824840698932617912025219930450699721398877046365,
  0.0000875929806464585519003991426948413177765384016472,
  0.0000929401858178341371667863418067094127307541377351,
  0.0000985335697696484674976329135611267909396711905609,
  0.0001043812033961267524414056775839236398397605678285,
  0.0001104913476358873787762733725235095222169775826075,
  0.0001168724569775170451646136908066947909359284171725,
  0.0001235331829579317139490988848356859391494532589618,
  0.0001304823776531504106421157108329639530888382586444,
  0.0001377290971610972794730068755717856537944796486076,
  0.0001452826050760356056997224727626496720694413609210,
  0.0001531523759542257489308945860421917545266975878153,
  0.0001613480987703871031700794454652400366813685870215,
  0.0001698796803645323154731489774417705543386041174765,
  0.0001787572488787300628451337953746267665801334405766,
  0.0001879911571833407131893105841972415835548127224662,
  0.0001975919862922571877036979801172886666913402646812,
  0.0002075705487666713060897943673680190454771171391258,
  0.0002179378921068738393305493426379626604843723728989,
  0.0002287053021315844246852150463491005367038196530909,
  0.0002398843063442954210517208619137486937664079174224,
  0.0002514866772861017071112459563670800999875862772490,
  0.0002635244358744763568752503947599815195484570366540,
  0.0002760098547274400746116896452062792743212568199345,
  0.0002889554614725602408695573486955982817731042477445,
  0.0003023740420402034207070514257206354000656033796341,
  0.0003162786439404532215359364273136414820662252696031,
  0.0003306825795230934685170167153155445378826964650062,
  0.0003455994292200447974864201151433310643546815996238,
  0.0003610430447696309562773380615124979675545409809284,
  0.0003770275524220393623519189743722727427598841377946,
  0.0003935673561253287952022142632400546494915901941708,
  0.0004106771406913255133474566622402725765929208100312,
  0.0004283718749407375852753810589145418574783787083069,
  0.0004466668148268058186703630547546516152544244238458,
  0.0004655775065367983700550093281593803311577528288179,
  0.0004851197895706449248470818503139854124035799270880,
  0.0005053097997959952630881683475279238998674672181446,
  0.0005261639724789760760043779630795840346051661062222,
  0.0005476990452899090803616235827013521638810686496264,
  0.0005699320612832427985037252530809222083323304447896,
  0.0005928803718509398392083755440640824187284430894400,
  0.0006165616396485511352313388661180603069932641265142,
  0.0006409938414931983747670392555122251563204408415780,
  0.0006661952712326758131312823871448921463403456127106,
  0.0006921845425848727748269509227101716927337696090048,
  0.0007189805919467084618010530391597621163897055912240,
  0.0007466026811717611781106173613541594033372788589215,
  0.0007750704003157647713058600134189729664061160838572,
  0.0008044036703491359834801244595683183413176877657085,
  0.0008346227458356875069406837215899398377606165358124,
  0.0008657482175766728575780050933997230673574010693814,
  0.0008978010152193007199479593964938444326524823642913,
  0.0009308024098288481884622270748877705977876146969302,
  0.0009647740164234943354704414263063679801520812067902,
  0.0009997377964709877859072235960614727254668365025499,
  0.0010357160603462544759892816344382872124212680130206,
  0.0010727314697490445265276253035892570235004255355979,
  0.0011108070400807101760346668676442416782184133260552,
  0.0011499661427792000011411960032071545865660111660188,
  0.0011902325076113482079954678491809138282498130839869,
  0.0012316302249215316143105529464467043598466441650111,
  0.0012741837478357610634816325610563919463775490374510,
  0.0013179178944202684255446674340292102066191200767307,
  0.0013628578497936450504293354978627045429890757962070,
  0.0014090291681915825526121081748644600628814846909603,
  0.0014564577749832621284392452634576342668934933064804,
  0.0015051699686384342435008211012835611621846252925084,
  0.0015551924226442264828266375454752438928026682075288,
  0.0016065521873707136365660043190084709426395540030038,
  0.0016592766918842807033183502883913646160919400276661,
  0.0017133937457078064374000082548245308654237088656707,
  0.0017689315405266923499484887701436255170006187626339,
  0.0018259186518397597016456213934958014631480117936830,
  0.0018843840405540350016315685291854343452098996513504,
  0.0019443570545224428574070811114387654727011879291844,
  0.0020058674300234237085811054392936401990600025530495,
  0.0020689452931814930274878791856186428447596557271188,
  0.0021336211613277579861160133866606111091142314167889,
  0.0021999259442994073754748162783238337407614840517227,
  0.0022678909456771907243503201991646039755879388827020,
  0.002337547863959903103120140773720307556925607552668,
  0.002408928793674893018510457808068607176076396504816,
  0.002482066226423612110359194685112906640377992773665,
  0.002556993051861227054925269829737298097581518748618,
  0.002633742558609316164240422462569172981269975474974,
  0.002712348435100675650479082203508458391962159587248,
  0.002792844770355263401218545772918975227444465028866,
  0.002875266054686311388524161877439657420846272465811,
  0.002959647180335641514620858248886413944213035391133,
  0.003046023442037223781950913884337249775410207110869,
  0.003134430537508020167963828326256044054923132109595,
  0.003224904567865162487179181823903689124127340035193,
  0.003317482037968517836894122697376816702606898105669,
  0.003412199856687700950203742730746873667591286611972,
  0.003509095337092598922437601601893035064441510682855,
  0.003608206196566480336202638645598729645649582458093,
  0.003709570556840767787315227915844372306600874035896,
  0.003813226943950560210195628624667646277258323789034,
  0.003919214288109999217816842761581767933063941331415,
  0.004027571923506581908914079483387422161983787524594,
  0.004138339588013531254573777862893750159923835262091,
  0.004251557422819344258070716797432723389492901411417,
  0.004367265971973647586280743386904643362890419198544,
  0.004485506181848500298371293768637057441018798108404,
  0.004606319400514293647801330437880665207622175547108,
  0.004729747377029408706818096697581063138841331671000,
  0.004855832260642803758323525448827350155503399682057,
  0.004984616599908715017732908511768983091803963913933,
  0.005116143341712666286628133513094122701536630464008,
  0.005250455830207995599813605722674011261809750069566,
  0.005387597805662119806841362522026640007870962505169,
  0.005527613403211771327039400025452051662544576252613,
  0.005670547151526455032240319611840180050718531169866,
  0.005816443971379387342282341024883411895100960679123,
  0.005965349174125194163287582160330129609399716756121,
  0.006117308460083659255889161652432860091240405273282,
  0.006272367916828829987984949662645628109775736572827,
  0.006430574017382803202077571476366494228242152467121,
  0.006591973618313530108483767514215490851050853195028,
  0.006756613957735995700158247114489179444715514066372,
  0.006924542653216145169905518107995821209210583602798,
  0.007095807699576947193506987625191416563492251927358,
  0.007270457466606001719760935972895502005009634908334,
  0.007448540696664118077443204273442012473626738017549,
  0.007630106502194307766403075526473106134867966790562,
  0.007815204363130655241902040492335692470114980887687,
  0.008003884124206549324207542558259939960027271144740,
  0.008196195992161777565529257537022102615212480537036,
  0.008392190532848005979627970440710736641783696532812,
  0.008591918668232186981669694884386953647077409720099,
  0.008795431673297459192811416603610021964749438260276,
  0.009002781172841123931099902951727525834630090702307,
  0.009214019138169304732891793616523380403128668912286,
  0.009429197883687918122353459157285469857882697273269,
  0.009648370063389606065707306962990976357134330521818,
  0.009871588667236303106635694548431660008992291854609,
  0.010098907017437134074358049703405331874641470997176,
  0.010330378764621361480931598876183169496309768224722,
  0.010566057883906125273709888082210029606544154799666,
  0.010805998670858741476894423499253311901145915424506,
  0.011050255737353350436853004250089009526615605526800,
  0.011298884007321729873326435834912273479343357620244,
  0.011551938712398112726630813177832559823704416695755,
  0.011809475387457874873169484639245034994188203633063,
  0.012071549866049983151539548006984227842642246077886,
  0.012338218275723119792654930842506154308328173347743,
  0.012609537033245425272876610855543714026212762673976,
  0.012885562839717827802269375146774597896625964567858,
  0.013166352675580954113788992218751954794161155951768,
  0.013451963795515642926307433930517236983197069019717,
  0.013742453723237109407641020003510396151814996209740,
  0.014037880246182836155763992583740830200805627627876,
  0.014338301410094293639650000827770538310987359766722,
  0.014643775513492620688045552759807083141277378597655,
  0.014954361102048423477181638671886382931616650584200,
  0.015270116962845879539093527612332511843438940388724,
  0.015591102118541361582849848577785479689233543376764,
  0.015917375821416824383483249696960740626279113934309,
  0.016248997547328226639548598146405632914432759472439,
  0.016586026989549288521685521469175935623639801958903,
  0.016928524052510914622899694911562102316729351926353,
  0.017276548845436641167968860298839910724255305085855,
  0.017630161675874495635792684129004754479789060095601,
  0.017989423043125686385910925958342613832281957503139,
  0.018354393631570569449988652292368480013392856555007,
  0.018725134303892369341895879455429375604454546685354,
  0.019101706094199160547089272417612713317808925575032,
  0.019484170201044646264247246613078847750101123016244,
  0.019872587980348300980346547841005942992806937501942,
  0.020267020938215473555348319482408378126433141255565,
  0.020667530723658077665058723174921669107448977307473,
  0.021074179121216526691144080441340253735513113942674,
  0.021487028043483600446243895007172333197423314607986,
  0.021906139523530961470100721930193790034351692758948,
  0.022331575707239069020013653546965574681097845158557,
  0.022763398845531269296061650411253255179407509826660,
  0.023201671286512870878716230451767154936229409888899,
  0.023646455467516044803898300124282166786414669557424,
  0.024097813907051419148408608952614348216934577025150,
  0.024555809196667268437105480612444493631409097716765,
  0.025020503992717228602303119124399166361635363579466,
  0.025491961008037498615664229956492084643838308289671,
  0.025970243003534520263369970007941030457915674721598,
  0.026455412779684157836542512811655626601055607750101,
  0.026947533167943429750714512136837579749195807611420,
  0.027446667022075874280502207204374270462899232748029,
  0.027952877209391661688438055759475935635066526749933,
  0.028466226601903595030027792952017293223563974896039,
  0.028986778067400171820372345492446085699992455913027,
  0.029514594460436908540980711700756969379565476417048,
  0.030049738613247159638529882952680312250524710101058,
  0.030592273326573692210130305777100228296216603915498,
  0.031142261360422306971956029246162941434470878038128,
  0.031699765424738825359724455010397560387820912751146,
  0.032264848170010791700293118605525736592951575563361,
  0.032837572177795268313420066051155882411170559823205,
  0.033417999951174130141361882761059837953350417600927,
  0.034006193905138294051326441010615186492566706619101,
  0.034602216356902346301742120932709404899571434041837,
  0.035206129516151059797760376729996696725031743934093,
  0.035817995475219320674308978543661795265329724428407,
  0.036437876199207011426323719638271848212068978893843,
  0.037065833516030425245504817121497341332833941233017,
  0.037701929106411813411105523630349292627758564747814,
  0.038346224493808694508939937707749059685328096895513,
  0.038998781034284580908114001568034343612609436265080,
  0.039659659906322804299105127576115692816669106680864,
  0.040328922100585148179959834413880454090780055396152,
  0.041006628409617020959817882970171680738977662822861,
  0.041692839417500928821036530806734479596916960892503,
  0.042387615489460032633272198885798551902580281460059,
  0.043091016761413598035436761045580238576208773037365,
  0.043803103129486172285008169404575979313300031298598,
  0.044523934239472345609338670932690359749540503298573,
  0.045253569476258978571042458449789880222672266959515,
  0.045992067953206800370011217412866395018363783456002,
  0.046739488501493306038936043575697759282439574025049,
  0.047495889659418903138328745160963981060038381316905,
  0.048261329661678280811944858777676072243063181634186,
  0.049035866428598995915317511338313487161561499134732,
  0.049819557555349292370014029116992329139236337093648,
  0.050612460301117190915523459638148541953222950723818,
  0.051414631578262907020772769026714333272270642870046,
  0.052226127941446674869657777108168071417768646970604,
  0.053047005576734075041275995387742346333247856081233,
  0.053877320290680982757488434962711363950830877372972,
  0.054717127499400272359857176626345298890687365828952,
  0.055566482217612431996864195691557085043476130137495,
  0.056425439047682260342694953354903558100561538755279,
  0.057294052168643834522972169622796525695216283720079,
  0.058172375325215955282982757551823262552725263766962,
  0.059060461816810291792616068930731772499098442005100,
  0.059958364486534464332019402792392859882009707225612,
  0.060866135710192318435608141067114284904015246517189,
  0.061783827385283658882412669729904546496529167460116,
  0.062711490920005726200813767905833804191687564525817,
  0.063649177222258712098668077487178731642119157807066,
  0.064596936688657623428958307826519372989262375091325,
  0.065554819193552816949871377673873032544296250436300,
  0.066522874078061539230216442442952403131528667404604,
  0.067501150139112817580103254853864164735431913790542,
  0.068489695618508058846726495404866801147281302130318,
  0.069488558191999723300020365435127564289711879278360,
  0.070497784958390450637098900492173304980974555037032,
  0.071517422428655024352184802448780679165995321056934,
  0.072547516515087569344723711994085297075661577218724,
  0.073588112520476385667321555227280607145092433508614,
  0.074639255127308828741941111940729014197859316753767,
  0.075700988387008653192535039972499817862441906430599,
  0.076773355709208243650236675875247258552875760829774,
  0.077856399851058161478815138613567178252569220913642,
  0.078950162906576441338945390434847991590656638541638,
  0.080054686296040075855746362400475925799919199439949,
  0.081170010755421130370983908079237229472941699101071,
  0.082296176325869932845488364229586137532533116527990,
  0.083433222343247786425054056585216679446492654814838,
  0.084581187427711653990914092332410471716385894003791,
  0.085740109473353265180552470621708172196835777295137,
  0.086910025637895096883053066858790880704065482421098,
  0.088090972332445678082510962098968136405906010046570,
  0.089282985211316669140560299084317378711670476378850,
  0.090486099161904164172314976771306774137173208424548,
  0.091700348294636663076682203878579362954646042542015,
  0.092925765932992157030001156169441784379254557806877,
  0.094162384603586767839386618583404038129689725464314,
  0.095410236026337377477328513180437133056859422183305,
  0.096669351104700679380522711600776712999738516869384,
  0.097939759915991077692299705999022685927866177001774,
  0.099221491701779854558292784854449414143264975284589,
  0.100514574858378018848268416738438877457089356162759,
  0.101819036927405242272656319898349412491464131686354,
  0.103134904586447280789799280260836807737136855951985,
  0.104462203639804270459034267644446217928929716021869,
  0.105800959009332277485365456418048982184645405680078,
  0.107151194725380472123853128268324678248006642728342,
  0.108512933917826285366285380912920763442502679460896,
  0.109886198807210895919796226821191021885419356262405,
  0.111271010695977382907627354754121410601185095122893,
  0.112667389959813866977194259432726042460114948425614,
  0.114075356039103949091212892241298384122104237729306,
  0.115494927430486742205282361082608722506857161501671,
  0.116926121678528776301624078584433061929779232796645,
  0.118368955367510041855479292852892208378048706168453,
  0.119823444113326420760005775141672992520018473542666,
  0.121289602555510737029640501438660820364537921227388,
  0.122767444349374642243267375503596478725044900714829,
  0.124256982158273532679814994420770121877974674977289,
  0.125758227645996676442985009907263405012280942593597,
  0.127271191469284709571760425980963662193955733505185,
  0.128795883270476640192456063460313135469572358146202,
  0.130332311670288479189847891762119249396892662878108,
  0.131880484260725594663057375825116837265025581748783,
  0.133440407598130865590279966826082535168131224585209,
  0.135012087196370687659246264029600808948439150760551,
  0.136595527520160861131806503933901243033072140389024,
  0.138190731978534366905752662380724218967375529543442,
  0.139797702918453012619657775340016444197124000513703,
  0.141416441618564905722039596019935600369687416981842,
  0.143046948283109684899665046250280711018679510363962,
  0.144689222035973415136620615634207945471287462781292,
  0.146343260914895024961396017494362367947509700905970,
  0.148009061865826137139373328611542452593951436469336,
  0.149686620737446116188684573082499877734600007569813,
  0.151375932275834127644492548708692894888806854783449,
  0.153076990119299974976648582812427284435605380833326,
  0.154789786793375450484861799107492709701029889381429,
  0.156514313705967906360639632937443667865425080181453,
  0.158250561142677721423176526280358179075563873765283,
  0.159998518262281307814108426208787305287280883758764,
  0.161758173092381270180828139092809249418465511232715,
  0.163529512525225297597263806988019827513704588413171,
  0.165312522313695335672230603090385744696163993060870,
  0.167107187067468552986420710564294540370975160736652,
  0.168913490249351582187718811389683612636693843238123,
  0.170731414171789481768911051015959006940028775447617,
  0.172560939993550829760255193272796006851226878944472,
  0.174402047716590325300225589461058169726525693402444,
  0.176254716183090238309630135035605869201902884074659,
  0.178118923072682011295970935615270277391787205646721,
  0.179994644899849280665298684483681329646086363583306,
  0.181881857011513547826962312765490752361778100468346,
  0.183780533584803692851804054296890288342966753221446,
  0.185690647625010485495870938912916575264122364498192,
  0.187612170963727210039130960879371793158869237513364,
  0.189545074257177481621665858171573180668586669736555,
  0.191489326984731292598175873215138665246410899072572,
  0.193444897447610287885328652016462255636635717094361,
  0.195411752767783228355605567515292742708069372912658,
  0.197389858887052561046069622654257559509895818585540,
  0.199379180566332974311256477423828346199025958150426,
  0.201379681385122775066659136681477040110027162340553,
  0.203391323741168883953647702968811491035486363551028,
  0.205414068850326202618870633465701572403733666347908,
  0.207447876746612065352074247788293829409563686625548,
  0.209492706282456445076819479381234897635571831128098,
  0.211548515129148541149847980466684886967032457537128,
  0.213615259777480333608041543852737487282587964072740,
  0.215692895538587645418322817269688038769242329380294,
  0.217781376544989210946857025190406221641344719047580,
  0.219880655751824205281028141761392083147641410985024,
  0.221990684938288645222468146756802865952032815807456,
  0.224111414709271028733595800372124289439577691270691,
  0.226242794497187535375441077919807469845166969944459,
  0.228384772564017065832846838590906672279246064153039,
  0.230537296003536353996385210093708953094704386691647,
  0.232700310743755340270514568924192968976044134196156,
  0.234873761549552949816719488679947135572729167047815,
  0.237057592025513374330776100813754682623913407142739,
  0.239251744618962910707090593044762307003840424131914,
  0.241456160623207364572553073803730088643871786489328,
  0.243670780180969981189875368818741646027449278117874,
  0.245895542288029820648336890868044822284777606634111,
  0.248130384797060448590695898662750796711052668215389,
  0.250375244421668767981229708072715706090269138214453,
  0.252630056740633771613985450240770579705772927474678,
  0.254894756202344949204930849789078874674097889524166,
  0.257169276129440037019405732354224606069083626478345,
  0.259453548723641752069734600881539488542575241544248,
  0.261747505070793106989741273850564013284078297628782,
  0.264051075146090855765904530974621145705005069280951,
  0.266364187819516574591724925347417274592095048034014,
  0.268686770861464836225269076803623102000528171371344,
  0.271018750948567890382562282334367859681793431380741,
  0.273360053669716216904264345549718599086128829346182,
  0.275710603532274272702642084228305565392797317092550,
  0.278070323968490707842999739130141256735474420708555,
  0.280439137342102280551195965929118367393508903523300,
  0.282816964955130655479405502851454175782201239643903,
  0.285203727054871224218604994949947752967313372412103,
  0.287599342841073041831089614109711210196468005254766,
  0.290003730473308928102353601221618265277389681455952,
  0.292416807078534737291563003731672821311037919666108,
  0.294838488758836755406253797500568488173536210176941,
  0.297268690599366139452412169683123997252523895237066,
  0.299707326676459268728308717420763375414805428443179,
  0.302154310065942834051897249168033067649448861028932,
  0.304609552851622446849740210633317919043827941000178,
  0.307072966133953506302726881807695425733710244438733,
  0.309544460038893019252695792691471746822000860115681,
  0.312023943726931024336791749552508176288520029109371,
  0.314511325402300228845253089757052818521728923121739,
  0.317006512322362424105545212634667323941983116741673,
  0.319509410807170202793473397837668246769664747589953,
  0.322019926249202459472191309725147694761122909366785,
  0.324537963123272113874866009567483220583947955504025,
  0.327063424996604454988081283889867070158976412186717,
  0.329596214539084462872691300437229359834974284247278,
  0.332136233533671424388522039510294446711209392011938,
  0.334683382886979118580714259724185911916707073673904,
  0.337237562640019807450170500266663847204252331737784,
  0.339798671979110228179973585375755462640117776459058,
  0.342366609246937743635147648956188468148301975489110,
  0.344941271953784769105992162475811545772748578098484,
  0.347522556788909554836583517855738716082039342321729,
  0.350110359632081365880943887596714424801531692115826,
  0.352704575565268063270744140443648599185195204427946,
  0.355305098884474053371038113187906005386551616003078,
  0.357911823111726535655098454399805078623311147916198,
  0.360524641007207942956492423216441488847773667143214,
  0.363143444581532432566522863229978480011931361092027,
  0.365768125108164251348355719415885994775989030143109,
  0.368398573135975763345715497222612692974455915678396,
  0.371034678501942894183969230007415652104777973325959,
  0.373676330343975712904610549643924052266307988833901,
  0.376323417113881838750325381523369615233761656875232,
  0.378975826590460327836547838990076613196236858885251,
  0.381633445892723662616125426341718835873468475146385,
  0.384296161493245435575678127337785919977433115013719,
  0.386963859231631287704569937923842511504939854161877,
  0.389636424328110631959066680534809982069256681662089,
  0.392313741397246662214019625230262046030677878083912,
  0.394995694461762119060913160203560951099234317016203,
  0.397682166966478255282799859989669678333738793588198,
  0.400373041792364415921799176734611524286528571733962,
  0.403068201270695620561563373895332564171389628994519,
  0.405767527197315508783345736115634832346999705578392,
  0.408470900847001983727791235944356441986072327874873,
  0.411178202987932863312876146750131382664060669935850,
  0.413889313896248823928933274252266862088661362609962,
  0.416604113370710897361609157736184573172951746258009,
  0.419322480747449758289914980955500085331637479059159,
  0.422044294914804016976068585505709014064370336401705,
  0.424769434328244709713201595122313201731546633934900,
  0.427497777025383158232647951007645515381075135007771,
  0.430229200641059348600664784387605477541860175459935,
  0.432963582422507960161090111579991694303270924899519,
  0.435700799244599155811438926340834201971061853671239,
  0.438440727625151226340900604008475703122724179387745,
  0.441183243740312163715041153248524151658483262706070,
  0.443928223440007221068941351636392612338862426298043,
  0.446675542263449500773014666440750969611965789779830,
  0.449425075454710596268634853993064064702258910321090,
  0.452176697978348298438537745265162625702235140938447,
  0.454930284535088363084106714870012830705820782019894,
  0.457685709577557322632253300066633467055563259669370,
  0.460442847326063312492593503858883007512672707945895,
  0.463201571784421870534708846616128997019680132360364,
  0.465961756755823656958962637289506913789791461098885,
  0.468723275858741031395889553516122452809460128431915,
  0.471486002542870414391642650649272366206465675300276,
  0.474249810105107351523196444833251101296981439798698,
  0.477014571705551190239574784082150075737542350002514,
  0.479780160383536272146681009641177438022712314882489,
  0.482546449073686536845513950587765323605100641950307,
  0.485313310621990427598589781042271210168173055170273,
  0.488080617801892984038963931396457507580071044102242,
  0.490848243330402002851839797975045977463362458178977,
  0.493616059884205143851615662343857386826901161165048,
  0.496383940115794856148384337656142613173098838834952,
  0.499151756669597997148160202024954022536637541821023,
  0.501919382198107015961036068603542492419928955897758,
  0.504686689378009572401410218957728789831826944829727,
  0.507453550926313463154486049412234676394899358049693,
  0.510219839616463727853318990358822561977287685117511,
  0.512985428294448809760425215917849924262457649997486,
  0.515750189894892648476803555166748898703018560201302,
  0.518513997457129585608357349350727633793534324699724,
  0.521276724141258968604110446483877547190539871568085,
  0.524038243244176343041037362710493086210208538901115,
  0.526798428215578129465291153383871002980319867639636,
  0.529557152673936687507406496141116992487327292054105,
  0.532314290422442677367746699933366532944436740330630,
  0.535069715464911636915893285129987169294179217980106,
  0.537823302021651701561462254734837374297764859061553,
  0.540574924545289403731365146006935935297741089678910,
  0.543324457736550499226985333559249030388034210220170,
  0.546071776559992778931058648363607387661137573701957,
  0.548816756259687836284958846751475848341516737293930,
  0.551559272374848773659099395991524296877275820612255,
  0.554299200755400844188561073659165798028938146328761,
  0.557036417577492039838909888420008305696729075100481,
  0.559770799358940651399335215612394522458139824540065,
  0.562502222974616841767352048992354484618924864992229,
  0.565230565671755290286798404877686798268453366065100,
  0.567955705085195983023931414494290985935629663598295,
  0.570677519252550241710085019044499914668362520940841,
  0.573395886629289102638390842263815426827048253741991,
  0.576110686103751176071066725747733137911338637390038,
  0.578821797012067136687123853249868617335939330064150,
  0.581529099152998016272208764055643558013927672125127,
  0.584232472802684491216654263884365167653000294421608,
  0.586931798729304379438436626104667435828610371005481,
  0.589626958207635584078200823265388475713471428266038,
  0.592317833033521744717200140010330321666261206411802,
  0.595004305538237880939086839796439048900765682983797,
  0.597686258602753337785980374769737953969322121916088,
  0.600363575671889368040933319465190017930743318337911,
  0.603036140768368712295430062076157488495060145838123,
  0.605703838506754564424321872662214080022566884986281,
  0.608366554107276337383874573658281164126531524853615,
  0.611024173409539672163452161009923386803763141114749,
  0.613676582886118161249674618476630384766238343124768,
  0.616323669656024287095389450356075947733692011166099,
  0.618965321498057105816030769992584347895222026674041,
  0.621601426864024236654284502777387307025544084321604,
  0.624231874891835748651644280584114005224010969856891,
  0.626856555418467567433477136770021519988068638907973,
  0.629475358992792057043507576783558511152226332856786,
  0.632088176888273464344901545600194921376688852083802,
  0.634694901115525946628961886812093994613448383996922,
  0.637295424434731936729255859556351400814804795572054,
  0.639889640367918634119056112403285575198468307884174,
  0.642477443211090445163416482144261283917960657678271,
  0.645058728046215230894007837524188454227251421901516,
  0.647633390753062256364852351043811531851698024510890,
  0.650201328020889771820026414624244537359882223540942,
  0.652762437359980192549829499733336152795747668262216,
  0.655316617113020881419285740275814088083292926326096,
  0.657863766466328575611477960489705553288790607988062,
  0.660403785460915537127308699562770640165025715752722,
  0.662936575003395545011918716110132929841023587813283,
  0.665462036876727886125133990432516779416052044495975,
  0.667980073750797540527808690274852305238877090633215,
  0.670490589192829797206526602162331753230335252410047,
  0.672993487677637575894454787365332676058016883258327,
  0.675488674597699771154746910242947181478271076878261,
  0.677976056273068975663208250447491823711479970890629,
  0.680455539961106980747304207308528253177999139884319,
  0.682927033866046493697273118192304574266289755561267,
  0.685390447148377553150259789366682080956172058999822,
  0.687845689934057165948102750831966932350551138971068,
  0.690292673323540731271691282579236624585194571556821,
  0.692731309400633860547587830316876002747476104762934,
  0.695161511241163244593746202499431511826463789823059,
  0.697583192921465262708436996268327178688962080333892,
  0.699996269526691071897646398778381734722610318544048,
  0.702400657158926958168910385890288789803531994745234,
  0.704796272945128775781395005050052247032686627587897,
  0.707183035044869344520594497148545824217798760356097,
  0.709560862657897719448804034070881632606491096476700,
  0.711929676031509292157000260869858743264525579291445,
  0.714289396467725727297357915771694434607202682907450,
  0.716639946330283783095735654450281400913871170653818,
  0.718981249051432109617437717665632140318206568619259,
  0.721313229138535163774730923196376897999471828628656,
  0.723635812180483425408275074652582725407904951965986,
  0.725948924853909144234095469025378854294994930719049,
  0.728252494929206893010258726149435986715921702371218,
  0.730546451276358247930265399118460511457424758455752,
  0.732830723870559962980594267645775393930916373521655,
  0.735105243797655050795069150210921125325902110475834,
  0.737369943259366228386014549759229420294227072525322,
  0.739624755578331232018770291927284293909730861785547,
  0.741869615202939551409304101337249203288947331784611,
  0.744104457711970179351663109131955177715222393365889,
  0.746329219819030018810124631181258353972550721882126,
  0.748543839376792635427446926196269911356128213510672,
  0.750748255381037089292909406955237692996159575868086,
  0.752942407974486625669223899186245317376086592857261,
  0.755126238450447050183280511320052864427270832952185,
  0.757299689256244659729485431075807031023955865803844,
  0.759462703996463646003614789906291046905295613308353,
  0.761615227435982934167153161409093327720753935846961,
  0.763757205502812464624558922080192530154833030055541,
  0.765888585290728971266404199627875710560422308729309,
  0.768009315061711354777531853243197134047967184192544,
  0.770119344248175794718971858238607916852358589014976,
  0.772218623455010789053142974809593778358655280952420,
  0.774307104461412354581677182730311961230757670619706,
  0.776384740222519666391958456147262512717412035927260,
  0.778451484870851458850152019533315113032967542462872,
  0.780507293717543554923180520618765102364428168871902,
  0.782552123253387934647925752211706170590436313374452,
  0.784585931149673797381129366534298427596266333652092,
  0.786608676258831116046352297031188508964513636448972,
  0.788620318614877224933340863318522959889972837659447,
  0.790620819433667025688743522576171653800974041849574,
  0.792610141112947438953930377345742440490104181414460,
  0.794588247232216771644394432484707257291930627087342,
  0.796555102552389712114671347983537744363364282905639,
  0.798510673015268707401824126784861334753589100927428,
  0.800454925742822518378334141828426819331413330263445,
  0.802387829036272789960869039120628206841130762486636,
  0.804309352374989514504129061087083424735877635501808,
  0.806219466415196307148195945703109711657033246778554,
  0.808118142988486452173037687234509247638221899531654,
  0.810005355100150719334701315516318670353913636416694,
  0.811881076927317988704029064384729722608212794353279,
  0.813745283816909761690369864964394130798097115925341,
  0.815597952283409674699774410538941830273474306597556,
  0.817439060006449170239744806727203993148773121055528,
  0.819268585828210518231088948984040993059971224552383,
  0.821086509750648417812281188610316387363306156761877,
  0.822892812932531447013579289435705459629024839263348,
  0.824687477686304664327769396909614255303836006939130,
  0.826470487474774702402736193011980172486295411586829,
  0.828241826907618729819171860907190750581534488767285,
  0.830001481737718692185891573791212694712719116241236,
  0.831749438857322278576823473719641820924436126234717,
  0.833485686294032093639360367062556332134574919818547,
  0.835210213206624549515138200892507290298970110618571,
  0.836923009880700025023351417187572715564394619166674,
  0.838624067724165872355507451291307105111193145216551,
  0.840313379262553883811315426917500122265399992430187,
  0.841990938134173862860626671388457547406048563530664,
  0.843656739085104975038603982505637632052490299094030,
  0.845310777964026584863379384365792054528712537218708,
  0.846953051716890315100334953749719288981320489636038,
  0.848583558381435094277960403980064399630312583018158,
  0.850202297081546987380342224659983555802875999486297,
  0.851809268021465633094247337619275781032624470456558,
  0.853404472479839138868193496066098756966927859610976,
  0.854987912803629312340753735970399191051560849239449,
  0.856559592401869134409720033173917464831868775414791,
  0.858119515739274405336942624174883162734974418251217,
  0.859667688329711520810152108237880750603107337121892,
  0.861204116729523359807543936539686864530427641853798,
  0.862728808530715290428239574019036337806044266494815,
  0.864241772354003323557014990092736594987719057406403,
  0.865743017841726467320185005579229878122025325022711,
  0.867232555650625357756732624496403521274955099285171,
  0.868710397444489262970359498561339179635462078772612,
  0.870176555886673579239994224858327007479981526457334,
  0.871631044632489958144520707147107791621951293831547,
  0.873073878321471223698375921415566938070220767203355,
  0.874505072569513257794717638917391277493142838498329,
  0.875924643960896050908787107758701615877895762270694,
  0.877332610040186133022805740567273957539885051574386,
  0.878728989304022617092372645245878589398814904877107,
  0.880113801192789104080203773178808978114580643737595,
  0.881487066082173714633714619087079236557497320539104,
  0.882848805274619527876146871731675321751993357271658,
  0.884199040990667722514634543581951017815354594319922,
  0.885537796360195729540965732355553782071070283978131,
  0.886865095413552719210200719739163192262863144048015,
  0.888180963072594757727343680101650587508535868313646,
  0.889485425141621981151731583261561122542910643837241,
  0.890778508298220145441707215145550585856735024715411,
  0.892060240084008922307700294000977314072133822998226,
  0.893330648895299320619477288399223287000261483130616,
  0.894589763973662622522671486819562866943140577816695,
  0.895837615396413232160613381416595961870310274535686,
  0.897074234067007842969998843830558215620745442193123,
  0.898299651705363336923317796121420637045353957457985,
  0.899513900838095835827685023228693225862826791575452,
  0.900717014788683330859439700915682621288329523621150,
  0.901909027667554321917489037901031863594093989953430,
  0.903089974362104903116946933141209119295934517578902,
  0.904259890526646734819447529378291827803164222704863,
  0.905418812572288346009085907667589528283614105996209,
  0.906566777656752213574945943414783320553507345185162,
  0.907703823674130067154511635770413862467466883472010,
  0.908829989244578869629016091920762770527058300898929,
  0.909945313703959924144253637599524074200080800560051,
  0.911049837093423558661054609565152008409343361458362,
  0.912143600148941838521184861386432821747430779086358,
  0.913226644290791756349763324124752741447124239170226,
  0.914299011612991346807464960027500182137558093569401,
  0.915360744872691171258058888059270985802140683246233,
  0.916411887479523614332678444772719392854907566491386,
  0.917452483484912430655276288005914702924338422781276,
  0.918482577571344975647815197551219320834004678943066,
  0.919502215041609549362901099507826695019025444962968,
  0.920511441808000276699979634564872435710288120721640,
  0.921510304381491941153273504595133198852718697869682,
  0.922498849860887182419896745146135835264568086209458,
  0.923477125921938460769783557557047596868471332595396,
  0.924445180806447183050128622326126967455703749563700,
  0.925403063311342376571041692173480627010737624908675,
  0.926350822777741287901331922512821268357880842192934,
  0.927288509079994273799186232094166195808312435474183,
  0.928216172614716341117587330270095453503470832539884,
  0.929133864289807681564391858932885715095984753482811,
  0.930041635513465535667980597207607140117990292774388,
  0.930939538183189708207383931069268227500901557994900,
  0.931827624674784044717017242448176737447274736233038,
  0.932705947831356165477027830377203474304783716279921,
  0.933574560952317739657305046645096441899438461244721,
  0.934433517782387568003135804308442914956523869862505,
  0.935282872500599727640142823373654701109312634171048,
  0.936122679709319017242511565037288636049169122627028,
  0.936952994423265924958724004612257653666752143918767,
  0.937773872058553325130342222891831928582231353029396,
  0.938585368421737092979227230973285666727729357129954,
  0.939387539698882809084476540361851458046777049276182,
  0.940180442444650707629985970883007670860763662906352,
  0.940964133571401004084682488661686512838438500865268,
  0.941738670338321719188055141222323927756936818365814,
  0.942504110340581096861671254839036018939961618683095,
  0.943260511498506693961063956424302240717560425974951,
  0.944007932046793199629988782587133604981636216543998,
  0.944746430523741021428957541550210119777327733040485,
  0.945476065760527654390661329067309640250459496701427,
  0.946196896870513827714991830595424020686699968701402,
  0.946908983238586401964563238954419761423791226962635,
  0.947612384510539967366727801114201448097419718539941,
  0.948307160582499071178963469193265520403083039107497,
  0.948993371590382979040182117029828319261022337177139,
  0.949671077899414851820040165586119545909219944603848,
  0.950340340093677195700894872423884307183330893319136,
  0.951001218965715419091885998431965656387390563734920,
  0.951653775506191305491060062292250940314671903104487,
  0.952298070893588186588894476369650707372241435252186,
  0.952934166483969574754495182878502658667166058766983,
  0.953562123800792988573676280361728151787931021106157,
  0.954182004524780679325691021456338204734670275571593,
  0.954793870483848940202239623270003303274968256065907,
  0.955397783643097653698257879067290595100428565958163,
  0.955993806094861705948673558989384813507433293380899,
  0.956582000048825869858638117238940162046649582399073,
  0.957162427822204731686579933948844117588829440176795,
  0.957735151829989208299706881394474263407048424436639,
  0.958300234575261174640275544989602439612179087248854,
  0.958857738639577693028043970753837058565529121961872,
  0.959407726673426307789869694222899771703783396084502,
  0.959950261386752840361470117047319687749475289898942,
  0.960485405539563091459019288299243030620434523582952,
  0.961013221932599828179627654507553914300007544086973,
  0.961533773398096404969972207047982706776436025103961,
  0.962047122790608338311561944240524064364933473250067,
  0.962553332977924125719497792795625729537100767251971,
  0.963052466832056570249285487863162420250804192388580,
  0.963544587220315842163457487188344373398944392249899,
  0.964029756996465479736630029992058969542084325278402,
  0.964508038991962501384335770043507915356161691710329,
  0.964979496007282771397696880875600833638364636420534,
  0.965444190803332731562894519387555506368590902283235,
  0.965902186092948580851591391047385651783065422974850,
  0.966353544532483955196101699875717833213585330442576,
  0.966798328713487129121283769548232845063770590111101,
  0.967236601154468730703938349588746744820592490173340,
  0.967668424292760930979986346453034425318902154841443,
  0.968093860476469038529899278069806209965648307241052,
  0.968512971956516399553756104992827666802576685392014,
  0.968925820878783473308855919558659746264486886057326,
  0.969332469276341922334941276825078330892551022692527,
  0.969732979061784526444651680517591621873566858744435,
  0.970127412019651699019653452158994057007193062498058,
  0.970515829798955353735752753386921152249898876983756,
  0.970898293905800839452910727582387286682191074424968,
  0.971274865696107630658104120544570624395545453314646,
  0.971645606368429430550011347707631519986607143444993,
  0.972010576956874313614089074041657386167718042496861,
  0.972369838324125504364207315870995245520210939904399,
  0.972723451154563358832031139701160089275744694914145,
  0.973071475947489085377100305088437897683270648073647,
  0.973413973010450711478314478530824064376360198041097,
  0.973751002452671773360451401853594367085567240527561,
  0.974082624178583175616516750303039259373720886065691,
  0.974408897881458638417150151422214520310766456623236,
  0.974729883037154120460906472387667488156561059611276,
  0.975045638897951576522818361328113617068383349415800,
  0.975356224486507379311954447240192916858722621402345,
  0.975661698589905706360349999172229461689012640233278,
  0.975962119753817163844236007416259169799194372372124,
  0.976257546276762890592358979996489603848185003790260,
  0.976548036204484357073692566069482763016802930980283,
  0.976833647324419045886211007781248045205838844048232,
  0.977114437160282172197730624853225402103374035432142,
  0.977390462966754574727123389144456285973787237326024,
  0.977661781724276880207345069157493845691671826652257,
  0.977928450133950016848460451993015772157357753922114,
  0.978190524612542125126830515360754965005811796366937,
  0.978448061287601887273369186822167440176295583304245,
  0.978701115992678270126673564165087726520656642379756,
  0.978949744262646649563146995749910990473384394473200,
  0.979194001329141258523105576500746688098854084575494,
  0.979433942116093874726290111917789970393455845200334,
  0.979669621235378638519068401123816830503690231775278,
  0.979901092982562865925641950296594668125358529002824,
  0.980128411332763696893364305451568339991007708145391,
  0.980351629936610393934292693037009023642865669478182,
  0.980570802116312081877646540842714530142117302726731,
  0.980785980861830695267108206383476619596871331087714,
  0.980997218827158876068900097048272474165369909297693,
  0.981204568326702540807188583396389978035250561739724,
  0.981408081331767813018330305115613046352922590279901,
  0.981607809467151994020372029559289263358216303467188,
  0.981803804007838222434470742462977897384787519462964,
  0.981996115875793450675792457441740060039972728855260,
  0.982184795636869344758097959507664307529885019112313,
  0.982369893497805692233596924473526893865132033209438,
  0.982551459303335881922556795726557987526373261982451,
  0.982729542533393998280239064027104497994990365091666,
  0.982904192300423052806493012374808583436507748072642,
  0.983075457346783854830094481892004178790789416397202,
  0.983243386042264004299841752885510820555284485933628,
  0.983408026381686469891516232485784509148949146804972,
  0.983569425982617196797922428523633505771757847532879,
  0.983727632083171170012015050337354371890224263427173,
  0.983882691539916340744110838347567139908759594726718,
  0.984034650825874805836712417839669870390600283243879,
  0.984183556028620612657717658975116588104899039320877,
  0.984329452848473544967759680388159819949281468830134,
  0.984472386596788228672960599974547948337455423747387,
  0.984612402194337880193158637477973359992129037494831,
  0.984749544169792004400186394277325988738190249930434,
  0.984883856658287333713371866486905877298463369535992,
  0.985015383400091284982267091488231016908196036086067,
  0.985144167739357196241676474551172649844496600317943,
  0.985270252622970591293181903302418936861158668329000,
  0.985393680599485706352198669562119334792377824452892,
  0.985514493818151499701628706231362942558981201891596,
  0.985632734028026352413719256613095356637109580801456,
  0.985748442577180655741929283202567276610507098588583,
  0.985861660411986468745426222137106249840076164737909,
  0.985972428076493418091085920516612577838016212475406,
  0.986080785711890000782183157238418232066936058668585,
  0.986186773056049439789804371375332353722741676210966,
  0.986290429443159232212684772084155627693399125964104,
  0.986391793803433519663797361354401270354350417541907,
  0.986490904662907401077562398398106964935558489317145,
  0.986587800143312299049796257269253126332408713388028,
  0.986682517962031482163105877302623183297393101894331,
  0.986775095432134837512820818176096310875872659964807,
  0.986865569462491979832036171673743955945076867890405,
  0.986953976557962776218049086115662750224589792889131,
  0.987040352819664358485379141751113586055786964608867,
  0.987124733945313688611475838122560342579153727534189,
  0.987207155229644736598781454227081024772555534971134,
  0.987287651564899324349520917796491541608037840412752,
  0.987366257441390683835759577537430827018730024525026,
  0.987443006948138772945074730170262701902418481251382,
  0.987517933773576387889640805314887093359622007226335,
  0.987591071206325106981489542191931392823923603495184,
  0.987662452136040096896879859226279692443074392447332,
  0.987732109054322809275649679800835396024412061117298,
  0.987800074055700592624525183721676166259238515948277,
  0.987866378838672242013883986613339388890885768583211,
  0.987931054706818506972512120814381357155240344272881,
  0.987994132569976576291418894560706359800939997446950,
  0.988055642945477557142592918888561234527298812070816,
  0.988115615959445964998368431470814565654790100348650,
  0.988174081348160240298354378606504198536851988206317,
  0.988231068459473307650051511229856374482999381237366,
  0.988286606254292193562599991745175469134576291134329,
  0.988340723308115719296681649711608635383908059972334,
  0.988393447812629286363433995680991529057360445996996,
  0.988444807577355773517173362454524756107197331792471,
  0.988494830031361565756499178898716438837815374707492,
  0.988543542225016737871560754736542365733106506693520,
  0.988590970831808417447387891825135539937118515309040,
  0.988637142150206354949570664502137295457010924203793,
  0.988682082105579731574455332565970789793380879923269,
  0.988725816252164238936518367438943608053622450962549,
  0.988768369775078468385689447053553295640153355834989,
  0.988809767492388651792004532150819086171750186916013,
  0.988850033857220799998858803996792845413433988833981,
  0.988889192959919289823965333132355758321781586673945,
  0.988927268530250955473472374696410742976499574464402,
  0.988964283939653745524010718365561712787578731986979,
  0.989000262203529012214092776403938527274533163497450,
  0.989035225983576505664529558573693632019847918793210,
  0.989069197590171151811537772925112229402212385303070,
  0.989102198984780699280052040603506155567347517635709,
  0.989134251782423327142421994906600276932642598930619,
  0.989165377254164312493059316278410060162239383464188,
  0.989195596329650864016519875540431681658682312234292,
  0.989224929599684235228694139986581027033593883916143,
  0.989253397318828238821889382638645840596662721141079,
  0.989281019408053291538198946960840237883610294408776,
  0.989307815457415127225173049077289828307266230390995,
  0.989333804728767324186868717612855107853659654387289,
  0.989359006158506801625232960744487774843679559158422,
  0.989383438360351448864768661133881939693006735873486,
  0.989407119628149060160791624455935917581271556910560,
  0.989430067938716757201496274746919077791667669555210,
  0.989452300954710090919638376417298647836118931350374,
  0.989473836027521023923995622036920415965394833893778,
  0.989494690200204004736911831652472076100132532781855,
  0.989514880210429355075152918149686014587596420072912,
  0.989534422493463201629944990671840619668842247171182,
  0.989553333185173194181329636945245348384745575576154,
  0.989571628125059262414724618941085458142521621291693,
  0.989589322859308674486652543337759727423407079189969,
  0.989606432643874671204797785736759945350508409805829,
  0.989622972447577960637648081025627727257240115862205,
  0.989638956955230369043722661938487502032445459019072,
  0.989654400570779955202513579884856668935645318400376,
  0.989669317420476906531482983284684455462117303534994,
  0.989683721356059546778464063572686358517933774730397,
  0.989697625957959796579292948574279364599934396620366,
  0.989711044538527439759130442651304401718226895752256,
  0.989723990145272559925388310354793720725678743180066,
  0.989736475564125523643124749605240018480451542963346,
  0.989748513322713898292888754043632919900012413722751,
  0.989760115693655704578948279138086251306233592082578,
  0.989771294697868415575314784953650899463296180346909,
  0.989782062107893126160669450657362037339515627627101,
  0.989792429451233328693910205632631980954522882860874,
  0.989802408013707742812296302019882711333308659735319,
  0.989812008842816659286810689415802758416445187277534,
  0.989821242751121269937154866204625373233419866559423,
  0.989830120319635467684526851022558229445661395882523,
  0.989838651901229612896829920554534759963318631412978,
  0.989846847624045774251069105413957808245473302412185,
  0.989854717394923964394300277527237350327930558639079,
  0.989862270902838902720526993124428214346205520351392,
  0.989869517622346849589357884289167036046911161741356,
  0.989876466817042068286050901115164314060850546741038,
  0.989883127543022482954835386309193305209064071582827,
  0.989889508652364112621223726627476490477783022417392,
  0.989895618796603873247558594322416076360160239432171,
  0.989901466430230351532502367086438873209060328809439,
  0.989907059814182165862833213658193290587269245862265,
  0.989912407019353541448099600857305158682223461598353,
  0.989917515930106738208797478006954930027860112295364,
  0.989922394247790981440234559902890153359467218268773,
  0.989927049494267556630665691650510130013048673418764,
  0.989931489015440741067225964250203315506783195696227,
  0.989935719984794256010332443710689383281473171308680,
  0.989939749406932934250326753304449426987452824432619,
  0.989943584121129308773024869820582638544639196551251,
  0.989947230804874839047445371008419363049196333600713,
  0.989950695977435502103305989222506145265634003053441,
  0.989953986003411486082001137303126687376417044450370,
  0.989957107096300734316880445910843850040029304885395,
  0.989960065322066098221012986244717809517402862352957,
  0.989962866602705867327611151658740331481918395574643,
  0.989965516719827454734366576531230249158978210332412,
  0.989968021318224025942681749907547451494841230907102,
  0.989970385909453868650830482070543624845435219039138,
  0.989972615875422310451217235774005427780992509483799,
  0.989974716471966000591004618519390360825762772815334,
  0.989976692832439380977423190247677846970532239440726,
  0.989978549971303180439161450548468367298381037074131,
  0.989980292787714774888561871039519743431841832613170,
  0.989981926069120264461240712159966368880136375737058,
  0.989983454494848126935640760168371419196960056025439,
  0.989984882639704314750470643212745910731558990718708,
  0.989986214977568670738655423790001662810724878741956,
  0.989987455884992545278115675463945396231315177699744,
  0.989988609644797504918324490881327941328721828448362,
  0.989989680449675029673207072782016757699700220373633,
  0.989990672405787103071715575632346299687594443740029,
  0.989991589536367605723630723403558935147721963570829,
  0.989992435785324429586239167217775203782433515078280,
  0.989993215020842236304070480100303065587506507852878,
  0.989993931038985788935541621242994280104412206740799,
  0.989994587567303792073975096457081493525746495990027,
  0.989995188268433180812990607877181720613592610116912,
  0.989995736743703804194815941561167833477148195225869,
  0.989996236536743453711856099791506001304590515397854,
  0.989996691137083192104273990691173973137321176514203,
  0.989997103983762942106884996157630096221781095370126,
  0.989997478468937298945006035438528430464492495678388,
  0.989997817941481534258823804204831860889367442390359,
  0.989998125710597762747295973078189384314318756158551,
  0.989998405049421246163656193089011243563789706619244,
  0.989998659198626812363486125191684761720235527066098,
  0.989998891370035369901417889348641133521590603503660,
  0.989999104750220501192356641726331354030559156296547,
  0.989999302504115119496330192985020286217570034883208,
  0.989999487778618176951492411549004059444808500847134,
  0.989999663706201412566388813610833688023769483226259,
  0.989999833408516130489443319038866507380161355273658,
  0.990000000000000000000000000000000000000000000000000,
  0.990000000000000000000000000000000000000000000000000,
  0.990000000000000000000000000000000000000000000000000,
  0.990000000000000000000000000000000000000000000000000
};
